/**
 * @since 1.0.0
 */
import * as Context from "effect/Context";
import * as Effect from "effect/Effect";
import * as Exit from "effect/Exit";
import * as Fiber from "effect/Fiber";
import * as GlobalValue from "effect/GlobalValue";
import * as Layer from "effect/Layer";
import * as Option from "effect/Option";
import * as Runtime from "effect/Runtime";
import * as Scope from "effect/Scope";
import * as Stream from "effect/Stream";
import { unify } from "effect/Unify";
import * as HttpBody from "./HttpBody.js";
import * as ServerError from "./HttpServerError.js";
import * as ServerRequest from "./HttpServerRequest.js";
import * as ServerResponse from "./HttpServerResponse.js";
import * as internal from "./internal/httpApp.js";
import * as internalMiddleware from "./internal/httpMiddleware.js";
const handledSymbol = /*#__PURE__*/Symbol.for("@effect/platform/HttpApp/handled");
/**
 * @since 1.0.0
 * @category combinators
 */
export const toHandled = (self, handleResponse, middleware) => {
  const responded = Effect.withFiberRuntime(fiber => Effect.flatMap(self, response => {
    const request = Context.unsafeGet(fiber.currentContext, ServerRequest.HttpServerRequest);
    const handler = fiber.getFiberRef(currentPreResponseHandlers);
    if (handler._tag === "None") {
      ;
      request[handledSymbol] = true;
      return Effect.as(handleResponse(request, response), response);
    }
    return Effect.tap(handler.value(request, response), response => {
      ;
      request[handledSymbol] = true;
      return handleResponse(request, response);
    });
  }));
  const withErrorHandling = Effect.catchAllCause(responded, cause => Effect.withFiberRuntime(fiber => Effect.flatMap(ServerError.causeResponse(cause), ([response, cause]) => {
    const request = Context.unsafeGet(fiber.currentContext, ServerRequest.HttpServerRequest);
    const handler = fiber.getFiberRef(currentPreResponseHandlers);
    if (handler._tag === "None") {
      ;
      request[handledSymbol] = true;
      return Effect.zipRight(handleResponse(request, response), Effect.failCause(cause));
    }
    return Effect.zipRight(Effect.tap(handler.value(request, response), response => {
      ;
      request[handledSymbol] = true;
      return handleResponse(request, response);
    }), Effect.failCause(cause));
  })));
  const withMiddleware = unify(middleware === undefined ? internalMiddleware.tracer(withErrorHandling) : Effect.matchCauseEffect(middleware(internalMiddleware.tracer(withErrorHandling)), {
    onFailure: cause => Effect.withFiberRuntime(fiber => {
      const request = Context.unsafeGet(fiber.currentContext, ServerRequest.HttpServerRequest);
      if (handledSymbol in request) {
        return Effect.void;
      }
      return Effect.matchCauseEffect(ServerError.causeResponse(cause), {
        onFailure: _cause => handleResponse(request, ServerResponse.empty({
          status: 500
        })),
        onSuccess: ([response]) => handleResponse(request, response)
      });
    }),
    onSuccess: response => Effect.withFiberRuntime(fiber => {
      const request = Context.unsafeGet(fiber.currentContext, ServerRequest.HttpServerRequest);
      return handledSymbol in request ? Effect.void : handleResponse(request, response);
    })
  }));
  return Effect.uninterruptible(scoped(withMiddleware));
};
/**
 * If you want to finalize the http request scope elsewhere, you can use this
 * function to eject from the default scope closure.
 *
 * @since 1.0.0
 * @category Scope
 */
export const ejectDefaultScopeClose = scope => {
  ejectedScopes.add(scope);
};
/**
 * @since 1.0.0
 * @category Scope
 */
export const unsafeEjectStreamScope = response => {
  if (response.body._tag !== "Stream") {
    return response;
  }
  const fiber = Option.getOrThrow(Fiber.getCurrentFiber());
  const scope = Context.unsafeGet(fiber.currentContext, Scope.Scope);
  ejectDefaultScopeClose(scope);
  return ServerResponse.setBody(response, HttpBody.stream(Stream.ensuring(response.body.stream, Scope.close(scope, Exit.void)), response.body.contentType, response.body.contentLength));
};
const ejectedScopes = /*#__PURE__*/GlobalValue.globalValue("@effect/platform/HttpApp/ejectedScopes", () => new WeakSet());
const scoped = effect => Effect.flatMap(Scope.make(), scope => Effect.onExit(Scope.extend(effect, scope), exit => {
  if (ejectedScopes.has(scope)) {
    return Effect.void;
  }
  return Scope.close(scope, exit);
}));
/**
 * @since 1.0.0
 * @category fiber refs
 */
export const currentPreResponseHandlers = internal.currentPreResponseHandlers;
/**
 * @since 1.0.0
 * @category fiber refs
 */
export const appendPreResponseHandler = internal.appendPreResponseHandler;
/**
 * @since 1.0.0
 * @category fiber refs
 */
export const withPreResponseHandler = internal.withPreResponseHandler;
/**
 * @since 1.0.0
 * @category conversions
 */
export const toWebHandlerRuntime = runtime => {
  const httpRuntime = Runtime.make(runtime);
  const run = Runtime.runFork(httpRuntime);
  return (self, middleware) => {
    const resolveSymbol = Symbol.for("@effect/platform/HttpApp/resolve");
    const httpApp = toHandled(self, (request, response) => {
      response = unsafeEjectStreamScope(response);
      request[resolveSymbol](ServerResponse.toWeb(response, {
        withoutBody: request.method === "HEAD",
        runtime
      }));
      return Effect.void;
    }, middleware);
    return (request, context) => new Promise(resolve => {
      const contextMap = new Map(runtime.context.unsafeMap);
      if (Context.isContext(context)) {
        for (const [key, value] of context.unsafeMap) {
          contextMap.set(key, value);
        }
      }
      const httpServerRequest = ServerRequest.fromWeb(request);
      contextMap.set(ServerRequest.HttpServerRequest.key, httpServerRequest);
      httpServerRequest[resolveSymbol] = resolve;
      httpRuntime.context = Context.unsafeMake(contextMap);
      const fiber = run(httpApp);
      request.signal?.addEventListener("abort", () => {
        fiber.unsafeInterruptAsFork(ServerError.clientAbortFiberId);
      }, {
        once: true
      });
    });
  };
};
/**
 * @since 1.0.0
 * @category conversions
 */
export const toWebHandler = /*#__PURE__*/toWebHandlerRuntime(Runtime.defaultRuntime);
/**
 * @since 1.0.0
 * @category conversions
 */
export const toWebHandlerLayerWith = (layer, options) => {
  const scope = Effect.runSync(Scope.make());
  const dispose = () => Effect.runPromise(Scope.close(scope, Exit.void));
  let handlerCache;
  let handlerPromise;
  function handler(request, context) {
    if (handlerCache) {
      return handlerCache(request, context);
    }
    handlerPromise ??= Effect.gen(function* () {
      const runtime = yield* options.memoMap ? Layer.toRuntimeWithMemoMap(layer, options.memoMap) : Layer.toRuntime(layer);
      return handlerCache = toWebHandlerRuntime(runtime)(yield* options.toHandler(runtime), options.middleware);
    }).pipe(Scope.extend(scope), Effect.runPromise);
    return handlerPromise.then(f => f(request, context));
  }
  return {
    dispose,
    handler
  };
};
/**
 * @since 1.0.0
 * @category conversions
 */
export const toWebHandlerLayer = (self, layer, options) => toWebHandlerLayerWith(layer, {
  ...options,
  toHandler: () => Effect.succeed(self)
});
//# sourceMappingURL=HttpApp.js.map