import { ClassListMerger, ErrorMessage, ExpandedRouteConfig, ExtendObjectIf, FetchEsque, MaybePromise, ProgressGranularity, UploadThingError } from "@uploadthing/shared";
import * as uploadthing_types0 from "uploadthing/types";
import { AnyFileRoute, ClientUploadedFileData, EndpointArg, FileRouter, inferEndpointInput, inferEndpointOutput, inferErrorShape } from "uploadthing/types";

//#region src/types.d.ts
interface GenerateTypedHelpersOptions {
  /**
   * URL to the UploadThing API endpoint
   * @example "/api/uploadthing"
   * @example "https://www.example.com/api/uploadthing"
   *
   * If relative, host will be inferred from either the `VERCEL_URL` environment variable or `window.location.origin`
   *
   * @default (VERCEL_URL ?? window.location.origin) + "/api/uploadthing"
   */
  url?: string | URL;
  /**
   * Provide a custom fetch implementation.
   * @default `globalThis.fetch`
   * @example
   * ```ts
   * fetch: (input, init) => {
   *   if (input.toString().startsWith(MY_SERVER_URL)) {
   *     // Include cookies in the request to your API
   *     return fetch(input, {
   *       ...init,
   *       credentials: "include",
   *     });
   *   }
   *
   *   return fetch(input, init);
   * }
   * ```
   */
  fetch?: FetchEsque | undefined;
}
type UseUploadthingProps<TFileRoute extends AnyFileRoute, TServerOutput = inferEndpointOutput<TFileRoute>> = {
  /**
   * Called when the upload is submitted and the server is about to be queried for presigned URLs
   * Can be used to modify the files before they are uploaded, e.g. renaming them
   */
  onBeforeUploadBegin?: ((files: File[]) => Promise<File[]> | File[]) | undefined;
  /**
   * Called when presigned URLs have been retrieved and the file upload is about to begin
   */
  onUploadBegin?: ((fileName: string) => void) | undefined;
  /**
   * Control how granular the upload progress is reported
   * - "all" - No filtering is applied, all progress events are reported
   * - "fine" - Progress is reported in increments of 1%
   * - "coarse" - Progress is reported in increments of 10%
   * @default "coarse"
   */
  uploadProgressGranularity?: ProgressGranularity | undefined;
  /**
   * Called continuously as the file is uploaded to the storage provider
   */
  onUploadProgress?: ((p: number) => void) | undefined;
  /**
   * This option has been moved to your serverside route config.
   * Please opt-in by setting `awaitServerData: false` in your route
   * config instead.
   * ### Example
   * ```ts
   * f(
   *   { image: { maxFileSize: "1MB" } },
   *   { awaitServerData: false }
   * ).middleware(...)
   * ```
   * @deprecated
   * @see https://docs.uploadthing.com/api-reference/server#route-options
   */
  skipPolling?: ErrorMessage<"This option has been moved to your serverside route config. Please use `awaitServerData` in your route config instead.">;
  /**
   * Called when the file uploads are completed
   * @remarks If `RouteOptions.awaitServerData` is `true`, this will be
   * called after the serverside `onUploadComplete` callback has finished
   */
  onClientUploadComplete?: ((res: ClientUploadedFileData<TServerOutput>[]) => MaybePromise<void>) | undefined;
  /**
   * Called if the upload fails
   */
  onUploadError?: ((e: UploadThingError<inferErrorShape<TFileRoute>>) => MaybePromise<void>) | undefined;
  /**
   * Set custom headers that'll get sent with requests
   * to your server
   */
  headers?: HeadersInit | (() => MaybePromise<HeadersInit>) | undefined;
  /**
   * An AbortSignal to cancel the upload
   * Calling `abort()` on the parent AbortController will cause the
   * upload to throw an `UploadAbortedError`. In a future version
   * the function will not throw in favor of an `onUploadAborted` callback.
   */
  signal?: AbortSignal | undefined;
};
type UploadthingComponentProps<TRouter extends FileRouter, TEndpoint extends keyof TRouter> = Omit<UseUploadthingProps<TRouter[TEndpoint]>,
/**
 * Signal is omitted, component has its own AbortController
 * If you need to control the interruption with more granularity,
 * create your own component and pass your own signal to
 * `useUploadThing`
 * @see https://github.com/pingdotgg/uploadthing/pull/838#discussion_r1624189818
 */
"signal"> & {
  /**
   * Called when the upload is aborted
   */
  onUploadAborted?: (() => MaybePromise<void>) | undefined;
  /**
   * The endpoint from your FileRouter to use for the upload
   */
  endpoint: EndpointArg<TRouter, TEndpoint>;
  /**
   * URL to the UploadThing API endpoint
   * @example URL { /api/uploadthing }
   * @example URL { https://www.example.com/api/uploadthing }
   *
   * If relative, host will be inferred from either the `VERCEL_URL` environment variable or `window.location.origin`
   *
   * @default (VERCEL_URL ?? window.location.origin) + "/api/uploadthing"
   */
  url?: string | URL;
  /**
   * Provide a custom fetch implementation.
   * @default `globalThis.fetch`
   * @example
   * ```ts
   * fetch: (input, init) => {
   *   if (input.toString().startsWith(MY_SERVER_URL)) {
   *     // Include cookies in the request to your API
   *     return fetch(input, {
   *       ...init,
   *       credentials: "include",
   *     });
   *   }
   *
   *   return fetch(input, init);
   * }
   * ```
   */
  fetch?: FetchEsque | undefined;
  config?: {
    mode?: "auto" | "manual";
    appendOnPaste?: boolean;
    /**
     * Override the default class name merger, with e.g. tailwind-merge
     * This may be required if you're customizing the component
     * appearance with additional TailwindCSS classes to ensure
     * classes are sorted and applied in the correct order
     */
    cn?: ClassListMerger;
  };
  disabled?: boolean;
  /**
   * Callback called when files are selected
   *
   * @param acceptedFiles - The files that were accepted.
   */
  onChange?: (files: File[]) => void;
} & ExtendObjectIf<inferEndpointInput<TRouter[TEndpoint]>, {
  /**
   * The input to the endpoint, as defined using `.input()` on the FileRouter endpoint
   * @see https://docs.uploadthing.com/api-reference/server#input
   */
  input: inferEndpointInput<TRouter[TEndpoint]>;
}>;
//#endregion
//#region src/use-uploadthing.d.ts
/**
 * @internal - This is an internal function. Use `generateReactHelpers` instead.
 * The actual hook we export for public usage is generated from `generateReactHelpers`
 * which has the URL and FileRouter generic pre-bound.
 */
declare function useUploadThingInternal<TRouter extends FileRouter, TEndpoint extends keyof TRouter>(url: URL, endpoint: EndpointArg<TRouter, TEndpoint>, fetch: FetchEsque, opts?: UseUploadthingProps<TRouter[TEndpoint]>): {
  readonly startUpload: (...args: undefined extends inferEndpointInput<TRouter[TEndpoint]> ? [files: File[], input?: undefined] : [files: File[], input: inferEndpointInput<TRouter[TEndpoint]>]) => Promise<uploadthing_types0.ClientUploadedFileData<uploadthing_types0.inferEndpointOutput<TRouter[TEndpoint]>>[] | undefined>;
  readonly isUploading: boolean;
  readonly routeConfig: ExpandedRouteConfig | undefined;
};
/** @internal - This is an internal function. Use `generateReactHelpers` instead. */
declare const __useUploadThingInternal: typeof useUploadThingInternal;
declare const generateReactHelpers: <TRouter extends FileRouter>(initOpts?: GenerateTypedHelpersOptions) => {
  /**
   * Get the config for a given endpoint outside of React context.
   * @remarks Can only be used if the NextSSRPlugin is used in the app.
   */
  readonly getRouteConfig: (slug: EndpointArg<TRouter, keyof TRouter>) => ExpandedRouteConfig;
  readonly uploadFiles: <TEndpoint extends keyof TRouter>(slug: EndpointArg<TRouter, TEndpoint>, opts: Omit<uploadthing_types0.UploadFilesOptions<TRouter[TEndpoint]>, keyof uploadthing_types0.GenerateUploaderOptions>) => Promise<uploadthing_types0.ClientUploadedFileData<uploadthing_types0.inferEndpointOutput<TRouter[TEndpoint]>>[]>;
  readonly createUpload: <TEndpoint extends keyof TRouter, TServerOutput = uploadthing_types0.inferEndpointOutput<TRouter[TEndpoint]>>(slug: EndpointArg<TRouter, TEndpoint>, opts: Omit<uploadthing_types0.CreateUploadOptions<TRouter[TEndpoint]>, keyof uploadthing_types0.GenerateUploaderOptions>) => Promise<{
    pauseUpload: (file?: File) => void;
    resumeUpload: (file?: File) => void;
    done: <T extends File | void = void>(file?: T | undefined) => Promise<T extends File ? uploadthing_types0.ClientUploadedFileData<TServerOutput> : uploadthing_types0.ClientUploadedFileData<TServerOutput>[]>;
  }>;
  readonly routeRegistry: uploadthing_types0.RouteRegistry<TRouter>;
  readonly useUploadThing: <TEndpoint extends keyof TRouter>(endpoint: EndpointArg<TRouter, TEndpoint>, opts?: UseUploadthingProps<TRouter[TEndpoint]>) => {
    readonly startUpload: (...args: undefined extends inferEndpointInput<TRouter[TEndpoint]> ? [files: File[], input?: inferEndpointInput<TRouter[TEndpoint]> & undefined] : [files: File[], input: inferEndpointInput<TRouter[TEndpoint]>]) => Promise<uploadthing_types0.ClientUploadedFileData<uploadthing_types0.inferEndpointOutput<TRouter[TEndpoint]>>[] | undefined>;
    readonly isUploading: boolean;
    readonly routeConfig: ExpandedRouteConfig | undefined;
  };
};
//#endregion
export { GenerateTypedHelpersOptions, UploadthingComponentProps, UseUploadthingProps, __useUploadThingInternal, generateReactHelpers };
//# sourceMappingURL=use-uploadthing-CkqJn3G-.d.ts.map