"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const fs_1 = __importDefault(require("fs"));
const path_1 = __importDefault(require("path"));
const process_1 = __importDefault(require("process"));
class ConfigFileSerializer {
    constructor() {
        this.map = new Map([
            ['config_files', 'App specific configuration files. This could be something like project_name.json'],
            ['repository_url', 'Url to the repository, code-narrator tries to extract this from project file'],
            ['builderPlugins', 'These are the plugins used when building documentation. You can create your own plugin. Checkout the code-narrator docs HowTo create a builder plugin'],
            ['gptSystemCommands', 'These are system commends send to GPT with every query'],
            ['readmeRoot', 'Indicates if the documentation should create a README file in root of project'],
            ['builders', 'Array of user defined documentations. See code-narrator How to create a user defined builder']
        ]);
    }
    async serialize(config) {
        const configContent = `
const ConfigurationBuilder = require("code-narrator/dist/src/documentation/plugins/builders/Configuration/ConfigurationBuilder");
const FilesBuilder = require("code-narrator/dist/src/documentation/plugins/builders/Files/FilesBuilder");
const FoldersBuilder = require("code-narrator/dist/src/documentation/plugins/builders/Folders/FoldersBuilder");
const UserDefinedBuilder = require("code-narrator/dist/src/documentation/plugins/builders/UserDefined/UserDefinedBuilder");

/**
 * You can find the documentation about code-narrator.config.js at
 * https://github.com/ingig/code-narrator/blob/master/docs/Configuration/code-narrator.config.js.md
 *
 * @type {ICodeNarratorConfig}
 */
const config = {
    ${this.generateConfigString(config)}
}
module.exports = config;
`;
        fs_1.default.writeFileSync(path_1.default.join(process_1.default.cwd(), 'code-narrator.config.js'), configContent);
        if (!fs_1.default.existsSync(path_1.default.join(process_1.default.cwd(), '.code-narrator'))) {
            fs_1.default.mkdirSync(path_1.default.join(process_1.default.cwd(), '.code-narrator'));
        }
        if (!fs_1.default.existsSync(path_1.default.join(process_1.default.cwd(), '.code-narrator/gpt_questions'))) {
            fs_1.default.mkdirSync(path_1.default.join(process_1.default.cwd(), '.code-narrator/gpt_questions'));
        }
        fs_1.default.copyFileSync(path_1.default.join(__dirname, 'README.liquid'), path_1.default.join(process_1.default.cwd(), '.code-narrator/gpt_questions/README.liquid'));
        fs_1.default.copyFileSync(path_1.default.join(__dirname, 'overview_readme.liquid'), path_1.default.join(process_1.default.cwd(), '.code-narrator/gpt_questions/overview_readme.liquid'));
        fs_1.default.copyFileSync(path_1.default.join(__dirname, 'howto_create_howto.liquid'), path_1.default.join(process_1.default.cwd(), '.code-narrator/gpt_questions/howto_create_howto.liquid'));
    }
    generateConfigString(configObject, indentLevel = 1) {
        let indent = '  '.repeat(indentLevel);
        let configString = '';
        configObject.aiService = undefined;
        for (const [key, value] of Object.entries(configObject)) {
            if (typeof value === 'object' && !Array.isArray(value)) {
                configString += this.getCommentByKey(key, indent);
                configString += `${indent}${key}: {\n`;
                configString += this.generateConfigString(value, indentLevel + 1);
                configString += `${indent}},\n`;
            }
            else if (Array.isArray(value)) {
                configString += this.getCommentByKey(key, indent);
                configString += `${indent}${key}: [\n`;
                configString += value
                    .map((item) => {
                    if (typeof item === 'object') {
                        return (`${indent}  {\n` +
                            this.generateConfigString(item, indentLevel + 2) +
                            `${indent}  },`);
                    }
                    else if (typeof item === 'function') {
                        return `${indent} ${item.name},`;
                    }
                    else {
                        return `${indent}  ${JSON.stringify(item)},`;
                    }
                })
                    .join('\n');
                configString += `\n${indent}],\n`;
            }
            else {
                configString += this.getCommentByKey(key, indent);
                configString += `${indent}${key}: ${JSON.stringify(value)},\n`;
            }
        }
        return configString;
    }
    getCommentByKey(key, indent) {
        let value = this.map.get(key);
        if (!value)
            return '';
        return `${indent}// ${value}\n`;
    }
}
exports.default = ConfigFileSerializer;
//# sourceMappingURL=ConfigFileSerializer.js.map
