"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const openai_1 = require("openai");
const promises_1 = require("timers/promises");
const ConfigHelper_1 = __importDefault(require("../config/ConfigHelper"));
const DefaultSettings_1 = __importDefault(require("../config/DefaultSettings"));
class OpenAIRepository {
    constructor() {
        this.models = new Map([
            ['gpt-3.5-turbo', 4096],
            ['gpt-4', 8192],
            ['gpt-4-32k', 32768]
        ]);
        this.retryStatuses = [429, 500, 503];
        const configuration = new openai_1.Configuration({
            apiKey: ConfigHelper_1.default.OpenAiKey
        });
        this.openai = new openai_1.OpenAIApi(configuration);
    }
    async query(text) {
        return this.queryQuestions([text]);
    }
    async queryQuestions(questions, errorCount = 0, model, assistantMessages) {
        let messages = [];
        try {
            await (0, promises_1.setTimeout)(1 * 500);
            let config = ConfigHelper_1.default.config;
            //config should only be undefined on first run
            if (!config)
                config = DefaultSettings_1.default.get();
            if (!model) {
                model = config.gptModel;
            }
            let messageLength = 0;
            for (let i = 0; config.gptSystemCommands && i < config.gptSystemCommands.length; i++) {
                messageLength += config.gptSystemCommands[i][i].length;
                messages.push({ role: 'system', content: config.gptSystemCommands[i].replace('{DocumentationType}', ConfigHelper_1.default.DocumentationType) });
            }
            let maxTokens = this.models.get(model) ?? 8192;
            for (let i = 0; i < questions.length; i++) {
                let q = questions[i];
                if (messageLength + q.length > maxTokens) {
                    let length = parseInt(((maxTokens - messageLength) / 1.20).toString());
                    q = q.substring(0, length);
                    console.warn(`Warning - Content to long: I had to trim a file, only using first ${length} character`);
                }
                messages.push({ role: 'user', content: q });
                messageLength += q.length;
            }
            for (let i = 0; assistantMessages && i < assistantMessages.length; i++) {
                messageLength += assistantMessages[i].length;
                messages.push({ role: 'assistant', content: assistantMessages[i] });
            }
            maxTokens = maxTokens - messageLength;
            if (maxTokens < 0) {
                console.error(`Message is to long (${messageLength}). Will not query gpt`);
                return { answer: '', requestMessages: messages };
            }
            const completion = await this.openai.createChatCompletion({
                model: model,
                messages: messages,
                temperature: 0.1,
                max_tokens: parseInt(maxTokens.toString()),
                top_p: 1,
                frequency_penalty: 0,
                presence_penalty: 0
            });
            if (completion.data.choices.length == 0) {
                throw new Error(`Did not get answer. ChatGPT is down. Run again. `);
            }
            let response = {
                answer: completion.data.choices[0].message.content,
                requestMessages: messages
            };
            return response;
        }
        catch (e) {
            let message = e.response?.data?.error?.message ?? '';
            if (message.indexOf('You exceeded your current quota') != -1) {
                throw new Error(message);
            }
            if (message.indexOf('The model') != -1 && message.indexOf('does not exist') != -1) {
                console.warn(`You don't have access to ${DefaultSettings_1.default.gptModel}, downgrading to gtp-3.5-turbo`);
                let model = 'gpt-3.5-turbo';
                DefaultSettings_1.default.gptModel = model;
                return this.queryQuestions(questions, errorCount, model, assistantMessages);
            }
            if (e && this.retryStatuses.includes(e.response?.status) && errorCount < 3) {
                console.log('Sleep 3 sec');
                await (0, promises_1.setTimeout)(3 * 1000);
                return this.queryQuestions(questions, ++errorCount, model);
            }
            console.error('OpenAI error:', message);
            console.error('Error doing OpenAI query:', questions);
            return { answer: '', requestMessages: messages };
        }
    }
}
exports.default = OpenAIRepository;
//# sourceMappingURL=OpenAIRepository.js.map
